(*  Copyright (c) 2001 Anthony L Shipman *)


signature COUNTER =
sig
    type Counter

    val new:	int -> Counter 
    val incr:	Counter -> int -> unit
    val get:	Counter -> int
    val getEvt:	Counter -> int CML.event

end



structure Counter: COUNTER =
struct

    datatype Request = 
	    ReqIsIncr of int
	|   ReqIsGet of Reply CML.chan

    and Reply =
	    ReplyIsCount of int

    and Counter = Counter of {
    		    req_chan:	Request CML.chan
		    }


    fun delay n = CML.sync (CML.timeOutEvt (Time.fromSeconds n))


    fun new init =
    let
	val req_chan = CML.channel()

	fun counter() =
	let
	    fun loop count =
	    (
		case CML.recv req_chan of
		  ReqIsIncr n => loop (count + n)

		| ReqIsGet rpl_chan => 
		let
		    fun reply() =
		    (
			delay 0;
		    	CML.send(rpl_chan, ReplyIsCount count)
		    )
		in
		    CML.spawn reply;
		    loop count
		end
	    )
	in
	    loop init
	end

	val thread = CML.spawn counter
    in
	Counter
	{
	    req_chan = req_chan
	}
    end


    fun incr (Counter {req_chan, ...}) n =
    (
	CML.send(req_chan, ReqIsIncr n)
    )


    fun getEvt (Counter {req_chan, ...}) =
    let
	fun send() =
	let
	    val rpl_chan = CML.channel()

	    fun recv (ReplyIsCount n) = n
	    fun sender() = CML.send(req_chan, ReqIsGet rpl_chan)
	in
	    CML.spawn sender;
	    CML.wrap(CML.recvEvt rpl_chan, recv)
	end
    in
	CML.guard send
    end

    fun get counter = CML.sync(getEvt counter)

end






structure Main =
struct
    fun toErr msg = TextIO.output(TextIO.stdErr, msg)


    fun run() =
    let
	val obj = Counter.new 0

	fun time_out t = CML.wrap(
			    CML.timeOutEvt(Time.fromSeconds t),
			    fn () => ~1)
    in
	Counter.incr obj 3;
	Counter.incr obj ~1;

	let
	    val c = CML.select [Counter.getEvt obj, time_out 1]
	in
	    print(concat["The counter's value is ",
	    	         Int.toString c, "\n"])
	end
    end



    fun main(arg0, argv) =
    let
    in
	RunCML.doit(run, NONE);
        OS.Process.success
    end
    handle
      x =>
    (
	toErr(concat["Uncaught exception: ", exnMessage x, " from\n"]);
	app (fn s => (print "\t"; print s; print "\n")) (SMLofNJ.exnHistory x);
	OS.Process.failure
    )

    val _ = SMLofNJ.exportFn("counter4", main)
end



