(*  Copyright (c) 2001 Anthony L Shipman *)

(* $Id: abort.sml,v 1.9 2001/10/20 21:00:31 felix Exp $ *)

(*  An abort value is represented by an IVar (SyncVar.ivar). This can
    be set like a flag, tested to see if it is set and an event is
    available for when it becomes set.  

    Setting the IVar is useful if it is discovered that a socket
    connection has broken.  It is important that each connection has
    its own IVar so that if the IVar is set it only forces an abort on
    the one connection.

    For scalability, we only have a single timer implemented in a
    singleton manager which maps from times to IVars.  The manager counts
    time in seconds since the start of the manager.  It maintains a list
    of IVars for each future second where a time-out is needed.

    IVars that have expired are just let go. They will be collected if
    their connection has gone away.
@#34567890123456789012345678901234567890123456789012345678901234567890

*)

signature ABORT =
sig
    type Abort

    (*  The arg is the time-out in seconds. *)
    val create:	    int -> Abort

    (*	This never times out. *)
    val never:	    unit -> Abort

    (*	This returns an abort event for synchronising. *)
    val evt:	    Abort -> unit CML.event

    (*	This tests if the event has occurred. *)
    val aborted:    Abort -> bool

    (*	This forces the abort to happen early even if it is the 'never'
	condition.
    *)
    val force:	    Abort -> unit
end


structure Abort: ABORT =
struct

    structure TF     = TextFrag

(*------------------------------------------------------------------------------*)

    datatype Abort = Abort of unit SyncVar.ivar


(*------------------------------------------------------------------------------*)

    (*	This maintains a well-balance tree to map from future times to aborts.
	We will only retain weak references to the aborts so that they drop
	out of the map when the client is no longer interested in them.

	Since the key is fixed at type int I use as a measure of time
	the number of seconds since the server started. This should fit
	into 30 bits!

    *)
    structure Map = IntRedBlackMap

    datatype Request = 
	    Add of int * Abort	    	(* (timeout, force) -> abort *)

    datatype State = State of {
	time:	    int,		(* seconds since startup *)
    	live:	    (Abort list) Map.map(* waiting to expire *)
	}


    fun server ch () =
    let
    	val start = Time.now()

	fun toTime secs = Time.fromSeconds(LargeInt.fromInt secs)
	fun trunc  time = Int.fromLarge(Time.toSeconds time)


	fun loop (state as State {time, ...}) =
	let
	    fun request (Add (delay, abort)) = add delay abort state


	    (*	If the timing drifts off it won't hurt if this 
		event is for a time in the past. It will be immediately
		enabled.
	    *)
	    val time_evt = CML.atTimeEvt(Time.+(start, toTime(time+1)))

	    val new_state = CML.select[
		    CML.wrap(CML.recvEvt ch,
			MyProfile.timeIt "abort request" request),

		    CML.wrap(time_evt,
		    	(*MyProfile.timeIt "abort expire"*) (expire state))
		    ]
	in
	    loop new_state
	end


	and add delay abort (state as State {time, live}) =
	let
	    (*	Find out the end-time in seconds relative to
		the start time of the server, rounded to the
		nearest second.
	    *)
	    val now   = Time.now()
	    val since = Time.-(now, start)
	    val ends  = trunc(Time.+(
	    		    Time.+(since, toTime delay),
	    		    Time.fromMilliseconds 500
			    ))

	    val _ = Log.testInform Globals.TestTimeout Log.Debug
		    (fn()=>TF.L ["Abort add delay=", Int.toString delay,
		    		 " now= ", Time.fmt 6 now,
				 " ends=", Int.toString ends
				 ])

	    (* The insert operation will either insert or replace. *)
	    fun add_abort() =
	    (
		case Map.find(live, ends) of
		  NONE =>
		    let
			val new_live = Map.insert(live, ends, [abort])
		    in
			State {time=time, live=new_live}
		    end

		| SOME ab_list =>
		    let
			val new_live = Map.insert(live, ends, abort::ab_list)
		    in
			State {time=time, live=new_live}
		    end
	    )
	in
	    add_abort()
	end


	(*  This scans all of the live entries looking for aborts to
	    expire.
	*)
	and expire (state as State {time, live}) () =
	let
	    (*	Find out what the time really is. *)
	    val count = trunc(Time.-(Time.now(), start))

	    fun check_entry (at_time, ab_list, new_live) =
	    (
		if count >= at_time
		then
		(
		    Log.testInform Globals.TestTimeout Log.Debug
			(fn()=>TF.L ["Abort expiring, count=",
			             Int.toString count,
				     " live size=",
				     Int.toString(Map.numItems live)
				     ]);

		    (* Remove the entry and set all its aborts. *)
		    app set_ab ab_list;
		    new_live
		)
		else
		    (*	Put the entry back into the map. *)
		    Map.insert(new_live, at_time, ab_list)
	    )


	    and set_ab (Abort ivar) = (SyncVar.iPut(ivar, ()))
					    handle _ => ()

	    val new_live = Map.foldli check_entry Map.empty live
	in
	    State {time=count, live=new_live}
	end

    in
	loop (State {time = 0, live = Map.empty})
    end



    structure Mgr = Singleton(
    			    type input    = Request CML.chan
			    val  newInput = CML.channel
			    val  object   = server
			    )

(*------------------------------------------------------------------------------*)

    fun create delay =
    let
	fun run() =
	let
	    val abort = Abort (SyncVar.iVar())
	in
	    CML.send(Mgr.get(), Add(delay, abort));
	    abort
	end
    in
	MyProfile.timeIt "abort create" run ()
    end


    fun evt     (Abort ivar) = SyncVar.iGetEvt ivar
    fun aborted (Abort ivar) = isSome(SyncVar.iGetPoll ivar)

    fun force   (Abort ivar) = (SyncVar.iPut(ivar, ()))
    				handle _ => ()

    fun never() = Abort (SyncVar.iVar())

end
