(*  Copyright (c) 2001 Anthony L Shipman *)

(* $Id: files.sml,v 1.7 2001/09/06 20:08:16 felix Exp $ *)

(*  This contains utility functions for files.
    This is leaf module.  Don't add anything that logs error messages.

*)

signature FILES =
sig

    (*	Check that a path is absolute.
    *)
    val absPath:		string -> bool

    (*	Get the directory component of a path. If it is not present
	then it returns "."
    *)
    val dirName:		string -> string

    (*	Get the base component of a path.
    *)
    val baseName:		string -> string

    (*	Split a path into an extension and the rest. If there is
	no extension then return NONE. The rest will include any
	directory parts.
    *)
    val splitExt:		string -> (string * string option)

    (*	Append a file name to a directory.
    *)
    val appendFile:		string -> string -> string

    (*	Check that a disk path exists.
    *)
    val exists:			string -> bool

    (*	Check if a disk path is a directory. Symbolic links are followed.
    *)
    val isDir:			string -> bool

    (*	Check if a disk path is a regular file. Symbolic links are followed.
    *)
    val isReg:			string -> bool

    (*	Check if a disk path is a symbolic link.
    *)
    val isSym:			string -> bool

    (*	Check that a disk path is a directory that is
        accessible for reading and searching.
    *)
    val accessibleDir:		string -> bool

    (*	Check that a disk path is a directory in which we can create files.
    *)
    val canCreateInDir:		string -> bool

    (*	Check that a disk path is a regular readable file.
    	Symbolic links are followed.
    *)
    val readableReg:		string -> bool

    (*	Check that a disk path is a regular readable and writable file.
    	Symbolic links are followed.
    *)
    val writableReg:		string -> bool

    (*	Check that a disk path is a regular executable file.
	This does not check for readability. Scripts may need to
	be readable.
    	Symbolic links are followed.
    *)
    val execableReg:		string -> bool

end



structure Files: FILES =
struct

    structure FS = Posix.FileSys
    structure P  = OS.Path

(*------------------------------------------------------------------------------*)

    fun absPath path = P.isAbsolute path


    fun dirName path =
    let
	val {dir, file} = P.splitDirFile path
    in
	if dir = "" then "." else dir
    end


    fun baseName path =
    let
	val {dir, file} = P.splitDirFile path
    in
	file
    end


    fun splitExt path =
    let
	val {base, ext} = P.splitBaseExt path
    in
	(base, ext)
    end


    fun appendFile dir file = P.joinDirFile {dir=dir, file=file}


    fun exists path = FS.access(path, [])

    fun isDir  path = exists path andalso FS.ST.isDir(FS.stat path)
    fun isReg  path = exists path andalso FS.ST.isReg(FS.stat path)
    fun isSym  path = exists path andalso FS.ST.isLink(FS.stat path)

    fun accessibleDir path =
    (
	isDir path andalso FS.access(path, [FS.A_READ, FS.A_EXEC])
    )

    fun canCreateInDir path =
    (
	isDir path andalso FS.access(path, [FS.A_READ, FS.A_WRITE, FS.A_EXEC])
    )

    fun readableReg path =
    (
	isReg path andalso FS.access(path, [FS.A_READ])
    )

    fun writableReg path =
    (
	isReg path andalso FS.access(path, [FS.A_READ, FS.A_WRITE])
    )

    fun execableReg path =
    (
	isReg path andalso FS.access(path, [FS.A_EXEC])
    )

(*------------------------------------------------------------------------------*)

end
