(*  Copyright (c) 2001 Anthony L Shipman *)

(* $Id: finalise.sml,v 1.8 2002/01/19 16:01:42 felix Exp $ *)

(*  This manages objects that need to be finalised when the garbage
    collector takes them.

    It is generic over the different types of objects.

    We maintain a list of weak references to the files.  This will be
    scanned periodically to finalise them.  

    While the client holds onto an Holder the finaliser will not be
    run.  The client must remove the object if it finalises it itself.
    The references are given an integer key so that they can be removed
    again later.  There is no other protection against both the client
    and the manager both finalising the object.

@#345678901234567890123456789012345678901234567890123456789012345
*)

(*==============================================================================*)

signature FINALISE_TYPE =
sig
    type T

    val finalise:   T -> unit
    val name:	    string
end




signature FINALISER =
sig
    (*	This is the value that is shared between the client 
	and the manager.
    *)
    type Holder

    (*	This is the value in the holder that will be finalised.  *)
    type T

    val get:	Holder -> T

    (*	This adds a new T to the manager.  *)
    val	add:	T -> Holder

    val remove:	Holder -> unit
end



functor FinaliseFn(
    structure Type: FINALISE_TYPE
    ): FINALISER =
struct

    structure Sy = SyncVar
    structure TF = TextFrag
    structure G  = Globals
    structure W  = SMLofNJ.Weak

(*------------------------------------------------------------------------------*)

    type    T = Type.T

    type Pair = int * T

    (*	We use a ref on the Pairs to ensure that they are 
	copied by reference.
    *)
    type Holder = Pair ref


    fun key (holder: Holder) = #1(!holder)
    fun get (holder: Holder) = #2(!holder)

(*------------------------------------------------------------------------------*)

    datatype Req = 
	    ReqAdd of T * Holder Sy.ivar
	|   ReqRemove of Holder

    (*	When the holder is collected we should have the last
	strong ref to T which we finalise.
    *)
    type Wref  = int * T * (Holder W.weak)

    (*	This requires a linear scan of all held objects which
	shouldn't be a performance problem since GCs are
	infrequent.
    *)
    type State = int * Wref list



    fun server chan () =
    let
	val gc_port = SignalMgr.mkGcPort()

	fun loop state =
	(
	    loop(CML.select[
		CML.wrap(CML.recvEvt chan, handle_msg state),
		CML.wrap(SignalMgr.gcEvt gc_port, finalise state)
		])
	)
    in
	loop(0, [])
    end



    and handle_msg (state as (tag_cnt, wrefs)) msg : State =
    (

	case msg of
	  ReqAdd (value, rvar) => 
	    let
		val _ = Log.testInform G.TestFinalise Log.Debug
		    (fn()=>TF.L ["Finaliser ", Type.name, ": add"])

		val key = tag_cnt
		val holder = ref (key, value)
		val new_state = (tag_cnt+1, (key, value, W.weak holder)::wrefs)
	    in
		Sy.iPut(rvar, holder);
		new_state
	    end

	| ReqRemove holder => 
	    let
		val _ = Log.testInform G.TestFinalise Log.Debug
		    (fn()=>TF.L ["Finaliser ", Type.name, ": remove"])

		val k = key holder
		fun keep (key, _, _) = k = key
	    in
		(tag_cnt, List.filter keep wrefs)
	    end
    )


    and finalise (state as (tag_cnt, wrefs)) () : State =
    let
	val _ = Log.testInform G.TestFinalise Log.Debug
		    (fn()=>TF.L ["Finaliser ", Type.name, ": finalising"])

	(*  Test if this wref should be kept or finalised.
	*)
	fun keep (_, value, wref) =
	(
	    case W.strong wref of
	      NONE   => (Type.finalise value; false)
	    | SOME _ => true
	)
    in
	(tag_cnt, List.filter keep wrefs)
    end



(*------------------------------------------------------------------------------*)

    structure Mgr = Singleton(
    			    type input    = Req CML.chan
			    val  newInput = CML.channel
			    val  object   = server
			    )

    fun	add value =
    let
	val rvar = Sy.iVar()
    in
	CML.send(Mgr.get(), ReqAdd (value, rvar));
	Sy.iGet rvar
    end


    fun remove holder = CML.send(Mgr.get(), ReqRemove holder)

(*------------------------------------------------------------------------------*)

end
