(*  Copyright (c) 2001 Anthony L Shipman *)

(* $Id: textfrag.sml,v 1.8 2002/03/12 18:09:58 felix Exp $ *)

(*  This defines a type for efficiently representing a collection
    of text fragments.  It is based on the wseq type of ML Server Pages
    from Moscow ML. 

    The goal is to delay the concatenation of strings as much as possible.
    Building trees is much faster.

*)

signature TEXTFRAG =
sig

    datatype Text =
	    Empty
	|   Nl                (* a line break, perhaps CRLF *)
	|   WS                (* exactly one blank character *)
	|   S of string
	|   L of string list  (* concatenation of some strings *)
	|   C of Text list    (* concatenation of texts *)

    datatype LineSep = UseLf | UseCrLf


    (*	This maps each element in a list to a Text value returning
	the concatentation of the values.
    *)
    val lift:	('a -> Text) -> 'a list -> Text

    (*	This does the same as txMap but inserts a separator, given as
	a Text, between each value.
    *)
    val liftSep:  Text -> ('a -> Text) -> 'a list -> Text

    (*	This applies the function to each string piece. The function
	could be print() for example.
    *)
    val apply:	LineSep -> (string -> unit) -> Text -> unit

    (*	Calculate the length in characters of the text.
    *)
    val length:	LineSep -> Text -> int

    (*	This is like apply but it prints the prefix before
	each subsequent line.
    *)
    val appPrefix:  string -> (string -> unit) -> Text -> unit

    (*	Produce the string that the Text corresponds to.
    *)
    val toString:  LineSep -> Text -> string

end


structure TextFrag: TEXTFRAG =
struct

(*------------------------------------------------------------------------------*)

    datatype Text =
	    Empty
	|   Nl			(* a line break, perhaps CRLF *)
	|   WS			(* exactly one blank character *)
	|   S of string
	|   L of string list	(* concatenation of some strings *)
	|   C of Text list	(* concatenation of texts *)

    datatype LineSep = UseLf | UseCrLf

(*------------------------------------------------------------------------------*)

    fun lift cvt lst = C (map cvt lst)


    fun liftSep sep cvt lst =
    let
	fun loop (x1::x2::t) rslt = loop (x2::t) (sep::(cvt x1)::rslt)
	|   loop [x1]        rslt = C (rev ((cvt x1)::rslt))
	|   loop []          rslt = C (rev rslt)
    in
	loop lst []
    end


    (*	crlf is the string to apply in place of Nl. *)
    fun applyP crlf func Empty   = ()
    |	applyP crlf func Nl      = func crlf
    |	applyP crlf func WS      = func " "
    |	applyP crlf func (S s)   = func s
    |	applyP crlf func (L ss)  = app func ss
    |	applyP crlf func (C lst) = app (applyP crlf func) lst


    fun lsep UseLf   = "\n"
    |   lsep UseCrLf = "\r\n"

    fun apply sep func text = applyP (lsep sep) func text

    fun appPrefix prefix func text = applyP ("\n" ^ prefix) func text


    fun toString sep text =
    let
	(*  This pushes the list of strings of its first arg onto the
	    reversed accumulator in rslt.
	*)
	val nl = lsep sep 

	fun flatten Empty   rslt = ""::rslt
	|   flatten Nl      rslt = nl::rslt
	|   flatten WS      rslt = " "::rslt
	|   flatten (S s)   rslt = s::rslt
	|   flatten (L ss)  rslt = push ss rslt
	|   flatten (C lst) rslt =
	let
	    fun loop []     rv = rv
	    |   loop (h::t) rv = loop t (flatten h rv)
	in
	    loop lst rslt
	end

	and push []     rslt = rslt
	|   push (h::t) rslt = push t (h::rslt)

    in
	concat(rev(flatten text []))
    end



    fun length sep text =
    let
	(*  This is like toString but we accumulate a length.
	*)
	val nl = size(lsep sep)

	fun flatten Empty   rslt = rslt
	|   flatten Nl      rslt = nl + rslt
	|   flatten WS      rslt = 1 + rslt
	|   flatten (S s)   rslt = (size s) + rslt
	|   flatten (L ss)  rslt = push ss rslt
	|   flatten (C lst) rslt =
	let
	    fun loop []     rv = rv
	    |   loop (h::t) rv = loop t (flatten h rv)
	in
	    loop lst rslt
	end

	and push []     rslt = rslt
	|   push (h::t) rslt = push t ((size h) + rslt)

    in
	flatten text 0
    end

(*------------------------------------------------------------------------------*)

end
