(*  Copyright (c) 2001 Anthony L Shipman *)

(* $Id: listener.sml,v 1.19 2002/03/10 17:18:25 felix Exp $ *)

(*  This runs the sockets.

@#34567890123456789012345678901234567890123456789012345678901234567890
*)

signature LISTENER =
sig

    (*	This runs the listener in the thread of the caller, which is the
	main thread. This never returns but it may throw an exception.
    *)
    val	run:	unit -> unit

end


structure Listener: LISTENER =
struct

    open Common
    open Config

    structure TF = TextFrag
    structure G  = Globals

    structure S = Socket
    structure PF  = Posix.FileSys
    structure PIO = Posix.IO

(*------------------------------------------------------------------------------*)

    (*	These are the requests to the connection manager.
    *)

    (*	This is our kind of connected socket for TCP connections. 
	INetSock.inet is the specialised type for the internet address family.
    *)
    type    OurConn = S.active INetSock.stream_sock
    type    OurAddr = INetSock.sock_addr

    datatype ListenMsg = ConnDied

(*------------------------------------------------------------------------------*)

    fun run() =
    let
	val ServerConfig {
		conn_timeout: int option,
		max_clients:  int option,
		listen_host,
		listen_port,
		...
		} = getServerConfig()

	(*  Build an address for listening.
	*)
	val listen_addr =
	    case listen_host of
	      NONE => INetSock.any listen_port

	    | SOME host =>
	    (
		(*  The configuration is supposed to have validated
		    the host.
		*)
		case NetHostDB.getByName host of
		  NONE       => raise InternalError "invalid host"
		| SOME entry =>
		    INetSock.toAddr(NetHostDB.addr entry, listen_port)
	    )

	val listener = INetSock.TCP.socket()
    in
	(*  Doing these fixes the type of the socket as passive. *)
	Socket.Ctl.setREUSEADDR(listener, true);
	Socket.bind(listener, listen_addr);
	Socket.listen(listener, 9);
	FileIO.setCloseExec(Socket.pollDesc listener);

	serve listener max_clients conn_timeout
    end
    handle x => (Log.logExn x; raise FatalX)



    (*	Here we spawn a thread for each connection. We
	catch the termination of each thread so that we can count them.
	If there are too many connections then we just close the new
	socket giving the client a Connection Refused error.

	Although we could retain all of the thread ids and use
	joinEvt() to select on their termination there might be
	scalability issues in a large server when doing a select()
	and joinEvt() on many connections.

	Instead we have each connection tell us when it dies.
	The connection thread will stay live to the GC although we
	don't hold its thread_id.

	Exceptions in here should close the listener socket.
    *)
    and serve listener max_clients conn_timeout =
    let
	val lchan: ListenMsg CML.chan = CML.channel()

	fun loop num_connects =
	let
	    (*	If we have too many then we will refuse the new
		connection.  We require each connection thread to tell
		us when it dies.

		We won't log the connection refusals to avoid the log
		overflowing on a DOS attack.
	    *)
	    fun new_connect (conn, conn_addr) =
	    (
		if (isSome max_clients) andalso
			num_connects >= (valOf max_clients)
		then
		(
		    Socket.close conn;
		    num_connects
		)
		else
		(
		    FileIO.setCloseExec(Socket.pollDesc conn);

		    CML.spawn(MyProfile.timeIt "Listener connection"
			(connection lchan conn conn_addr conn_timeout));

		    num_connects+1
		)
	    )
	    handle x =>
		(
		    (Socket.close conn) handle _ => ();
		    Log.logExn x;
		    num_connects
		)

	    fun msg ConnDied = num_connects - 1

	    val new_num = CML.select[
	    	CML.wrap(S.acceptEvt listener, new_connect),
		CML.wrap(CML.recvEvt lchan, msg)
		]
	in
	    loop new_num
	end
    in
	loop 0
    end
    handle x =>
	(
	    Socket.close listener;
	    Log.logExn x;
	    raise FatalX
	)


    (*	This runs in a thread to handle a connection.

	The time limit applies both to the client and the handler.
	If the client doesn't complete sending its request before the
	time limit then we force the connection closed. Similarly the
	protocol module must complete its reponse before the time limit.

	The protocol module is called as a function. It returns when it is
	time to close the connection or the time limit has been exceeded.
    *)

    and connection lchan sock sock_addr conn_timeout () =
    let
	fun run() =
	let
	    val conn = MyProfile.timeIt "Listener setupConn"
			Connect.setupConn{
	    		    socket  = sock,
			    sock_addr = sock_addr,
			    timeout = conn_timeout
			    }
	in
	    MyProfile.timeIt "Listener talk" 
		HTTP_1_0.talk conn;

	    MyProfile.timeIt "Listener close" 
		Connect.close conn;

	    Log.testInform G.TestConnect Log.Debug
	    	(fn()=>TF.S "Connection closed");

	    MyProfile.timeIt "Listener release" 
		TmpFile.releasePort(Connect.getPort conn);

	    Log.testInform G.TestConnect Log.Debug
		(fn()=>TF.S "TmpFiles released")
	end

    in
	Log.inform Log.Info (fn()=>TF.C [TF.S "New connection from ",
	                            format_addr sock_addr]);

	MyProfile.timeIt "Listener run" 
	    run();

	Log.inform Log.Info (fn()=>TF.C [TF.S "End of connection from ",
				    format_addr sock_addr]);

	MyProfile.timeIt "Listener died" 
	    CML.send(lchan, ConnDied)
    end
    handle x =>
	let
	    (*	See also Connect.getPort *)
	    val (_, port) = INetSock.fromAddr sock_addr
	in
	    (
		Socket.close sock;
		TmpFile.releasePort port
	    ) handle _ => ();		(* being paranoid *)
	    Log.logExn x;
	    CML.send(lchan, ConnDied)
	end


    and format_addr sock_addr =
    let
	val (in_addr, port) = INetSock.fromAddr sock_addr
    in
	TF.L [NetHostDB.toString in_addr, ":", Int.toString port]
    end

(*------------------------------------------------------------------------------*)


end

