(*  Copyright (c) 2001 Anthony L Shipman *)

(* $Id: node_factory.sml,v 1.11 2001/09/20 20:27:01 felix Exp $ *)

(*  This creates store nodes of different kinds.

    The possible kinds are:
	directory	- backed by a disk directory
	built-in	- delegate to an internal handler
	cgi-bin		- delegate to a CGI script

@#34567890123456789012345678901234567890123456789012345678901234567890
*)

signature NODE_FACTORY =
sig

    (*	Create a new node.  The kind is obtained from the configuration.
    *)
    val create: Node.NodeCreator

end



structure NodeFactory: NODE_FACTORY =
struct

    structure TF  = TextFrag
    structure Cfg = Config

(*------------------------------------------------------------------------------*)

    structure DirNode = GenericNodeFn(
		    structure Handler = DirNodeHandler)

    structure CgiNode = GenericNodeFn(
		    structure Handler = CgiNodeHandler)

    structure SimpleBuiltinNode = GenericNodeFn(
		    structure Handler = SimpleBuiltinHandler)

(*------------------------------------------------------------------------------*)

    fun create {config, children, options} =
    let
	val Cfg.NodeConfig {path, kind, ...} = config

	val () = Log.testInform Globals.TestStoreBuild Log.Debug
		(fn() => TF.L ["Creating node ", Cfg.listToString path]);

	(*  This name is used for locating children from path segments.
	    See GenericNodeFn.forward_child.
	*)
	val node_name = if null path then "/" else List.last path
    in
	case kind of
	  Cfg.NodeIsDir {path} =>
	    DirNode.create {
		name     = node_name,
		arg      = path,
		config   = config,
		options  = options,
		factory  = create,
		children = children
		}

	| Cfg.NodeIsBuiltin {name} =>
	    SimpleBuiltinNode.create {
		name     = node_name,
		arg      = name,
		config   = config,
		options  = options,
		factory  = create,
		children = children
		}

	| Cfg.NodeIsScript {path} =>
	    CgiNode.create {
		name     = node_name,
		arg      = path,
		config   = config,
		options  = options,
		factory  = create,
		children = children
		}
    end

(*------------------------------------------------------------------------------*)

end
