(*  Copyright (c) 2001 Anthony L Shipman *)

(* $Id: resp_utils.sml,v 1.11 2002/01/19 16:02:11 felix Exp $ *)

(*  

@#34567890123456789012345678901234567890123456789012345678901234567890
*)

signature RESP_UTILS =
sig
    (*	This creates a dummy configuration for a built-in
	handler that we want to insert at some node path.
    *)
    val builtinConfig:	Config.NodePath -> string -> Config.NodeConfig

    (*	This returns a response back to the client's connection. It
	marks the end of the resource store's involvement with the request.
    *)
    val return:		HTTPMsg.Request -> HTTPMsg.Response -> unit

    (*	These functions generate useful responses. 
    *)
    val mkNotFound:	unit -> HTTPMsg.Response
    val mkServerFail:	unit -> HTTPMsg.Response
    val mkForbidden:	unit -> HTTPMsg.Response

    val mkHelloWorld:	unit -> HTTPMsg.Response

    (*	This makes a response out of some piece of HTML.
    *)
    val mkHTML:	HTTPStatus.Status -> TextFrag.Text -> HTTPMsg.Response
end


structure ResponseUtils: RESP_UTILS =
struct

    open Common

    structure TF     = TextFrag
    structure Req    = HTTPMsg
    structure Status = HTTPStatus
    structure Hdr    = HTTPHeader

(*------------------------------------------------------------------------------*)

    fun builtinConfig node_path name =
    (
	Config.NodeConfig {
	    path	= node_path,
	    kind	= Config.NodeIsBuiltin {name = name},
	    options	= [],
	    auth	= Config.NodeNoAuth
	    }
    )


    fun return req response =
    let
	val Req.Request {rvar, ...} = req
    in
	SyncVar.iPut(rvar, response)
    end


(*------------------------------------------------------------------------------*)

    and mkHelloWorld() : Req.Response =
    let
	val info = Entity.Info {
		    etype	= SOME (Entity.simpleType "text" "plain"),
		    encoding    = NONE,
		    length      = NONE,
		    last_mod    = NONE
		    }

	val entity = Entity.Entity {
		    info = info,
		    body = Entity.textProducer(TF.C
				[TF.S "hello world", TF.Nl])
		    }
    in
	Req.Response {
	    status  = Status.OK,
	    headers = [],
	    entity  = entity
	    }
    end

(*------------------------------------------------------------------------------*)

    and mkNotFound() =
    (
	mkHTML Status.NotFound 
	    (lines_to_text [
		"<html><body>", 
		"<em>Not Found</em>",
		"</body></html>"
		])
    )



    and mkServerFail() =
    (
	mkHTML Status.ServerFail 
	    (lines_to_text [
		"<html><body>", 
		"<em>Internal Server Error, Tough Luck</em>",
		"</body></html>"
		])
    )



    and mkForbidden() =
    (
	mkHTML Status.Forbidden 
	    (lines_to_text [
		"<html><body>", 
		"<em>Access Denied</em>",
		"</body></html>"
		])
    )


    and lines_to_text lst =
    (
	TF.C(map (fn l => TF.C [TF.S l, TF.Nl]) lst)
    )



    and mkHTML status frag =
    let
	val info = Entity.Info {
		    etype	= SOME (Entity.simpleType "text" "html"),
		    encoding    = NONE,
		    length      = NONE,
		    last_mod    = NONE
		    }

	val entity = Entity.Entity {
		    info = info,
		    body = Entity.textProducer frag
		    }
    in
	Req.Response {
	    status  = status,
	    headers = [],
	    entity  = entity
	    }
    end


(*------------------------------------------------------------------------------*)

end
