/**
 * Utility function for custom functions.
 * Cast a value as a Boolean,
 * unwrapping objects using their `valueOf()` methods.
 * Also accepts `'true'` and `'false'`.
 * Throws a `RangeError` for invalid inputs.
 */
export function asBoolean(value) {
    if (value && typeof value === 'object')
        value = value.valueOf();
    if (typeof value === 'boolean')
        return value;
    if (value && typeof value === 'object')
        value = String(value);
    if (value === 'true')
        return true;
    if (value === 'false')
        return false;
    throw new RangeError('Not a boolean');
}
/**
 * Utility function for custom functions.
 * Cast a value as a non-negative integer,
 * unwrapping objects using their `valueOf()` methods.
 * Also accepts JSON string reprentations of integers.
 * Throws a `RangeError` for invalid inputs.
 *
 * The default functions use this to validate _digit size options_.
 */
export function asPositiveInteger(value) {
    if (value && typeof value === 'object')
        value = value.valueOf();
    if (value && typeof value === 'object')
        value = String(value);
    if (typeof value === 'string' && /^(0|[1-9][0-9]*)$/.test(value)) {
        value = Number(value);
    }
    if (typeof value === 'number' && value >= 0 && Number.isInteger(value)) {
        return value;
    }
    throw new RangeError('Not a positive integer');
}
/**
 * Utility function for custom functions.
 * Cast a value as a string,
 * unwrapping objects using their `valueOf()` methods.
 * Throws a `RangeError` for invalid inputs.
 */
export function asString(value) {
    if (value && typeof value === 'object')
        value = value.valueOf();
    if (typeof value === 'string')
        return value;
    if (value && typeof value === 'object')
        return String(value);
    throw new RangeError('Not a string');
}
