import type { Message } from './data-model/types.ts';
import type { MessagePart } from './formatted-parts.ts';
import { type MessageValue } from './message-value.ts';
import type { MessageFunctionContext } from './resolve/function-context.ts';
/**
 * An MF2 function handler, for use in {@link MessageFormatOptions.functions}.
 *
 * @category Formatting
 */
export type MessageFunction<T extends string, P extends string = T> = (context: MessageFunctionContext, options: Record<string, unknown>, input?: unknown) => MessageValue<T, P>;
/**
 * @category Formatting
 * @typeParam T - The `type` used by custom message functions, if any.
 *   These extend the {@link DefaultFunctions | default functions}.
 * @typeParam P - The formatted-parts `type` used by any custom message values.
 */
export interface MessageFormatOptions<T extends string = never, P extends string = T> {
    /**
     * The bidi isolation strategy for message formatting,
     * i.e. how expression placeholders with different or unknown directionalities
     * are isolated from the rest of the formatted message.
     *
     * The default `'default'` strategy isolates all expression placeholders,
     * except when both the message and the placeholder are known to be left-to-right.
     *
     * The `'none'` strategy applies no isolation at all.
     */
    bidiIsolation?: 'default' | 'none';
    /**
     * Explicitly set the message's base direction.
     * If not set, the direction is detected from the primary locale.
     */
    dir?: 'ltr' | 'rtl' | 'auto';
    /**
     * If given multiple locales,
     * determines which algorithm to use when selecting between them;
     * the default for `Intl` formatters is `'best fit'`.
     *
     * The locale is resolved separately by each message function handler call.
     *
     * See: https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Intl#locale_negotiation
     */
    localeMatcher?: 'best fit' | 'lookup';
    /**
     * A set of custom functions to make available during message resolution.
     * Extends the {@link DefaultFunctions | default functions}.
     */
    functions?: Record<string, MessageFunction<T, P>>;
}
/**
 * A message formatter for that implements the
 * {@link https://www.unicode.org/reports/tr35/tr35-75/tr35-messageFormat.html#contents-of-part-9-messageformat | LDML 47 MessageFormat}
 * specification as well as the {@link https://github.com/tc39/proposal-intl-messageformat/ | TC39 Intl.MessageFormat proposal}.
 *
 * @category Formatting
 * @typeParam T - The `type` used by custom message functions, if any.
 *   These extend the {@link DefaultFunctions | default functions}.
 * @typeParam P - The formatted-parts `type` used by any custom message values.
 */
export declare class MessageFormat<T extends string = never, P extends string = T> {
    #private;
    constructor(locales: string | string[] | undefined, source: string | Message, options?: MessageFormatOptions<T, P>);
    /**
     * Format a message to a string.
     *
     * ```js
     * import { MessageFormat } from 'messageformat';
     * import { DraftFunctions } from 'messageformat/functions';
     *
     * const msg = 'Hello {$user.name}, today is {$date :date style=long}';
     * const mf = new MessageFormat('en', msg, { functions: DraftFunctions });
     * mf.format({ user: { name: 'Kat' }, date: new Date('2025-03-01') });
     * ```
     *
     * ```js
     * 'Hello Kat, today is March 1, 2025'
     * ```
     *
     * @param msgParams - Values that may be referenced by `$`-prefixed variable references.
     *   To refer to an inner property of an object value,
     *   use `.` as a separator; in case of conflict, the longest starting substring wins.
     * @param onError - Called in case of error.
     *   If not set, errors are by default logged as warnings.
     */
    format(msgParams?: Record<string, unknown>, onError?: (error: unknown) => void): string;
    /**
     * Format a message to a sequence of parts.
     *
     * ```js
     * import { MessageFormat } from 'messageformat';
     * import { DraftFunctions } from 'messageformat/functions';
     *
     * const msg = 'Hello {$user.name}, today is {$date :date style=long}';
     * const mf = new MessageFormat('en', msg, { functions: DraftFunctions });
     * mf.formatToParts({ user: { name: 'Kat' }, date: new Date('2025-03-01') });
     * ```
     *
     * ```js
     * [
     *   { type: 'text', value: 'Hello ' },
     *   { type: 'bidiIsolation', value: '\u2068' },
     *   { type: 'string', locale: 'en', value: 'Kat' },
     *   { type: 'bidiIsolation', value: '\u2069' },
     *   { type: 'text', value: ', today is ' },
     *   {
     *     type: 'datetime',
     *     dir: 'ltr',
     *     locale: 'en',
     *     parts: [
     *       { type: 'month', value: 'March' },
     *       { type: 'literal', value: ' ' },
     *       { type: 'day', value: '1' },
     *       { type: 'literal', value: ', ' },
     *       { type: 'year', value: '2025' }
     *     ]
     *   }
     * ]
     * ```
     *
     * @param msgParams - Values that may be referenced by `$`-prefixed variable references.
     *   To refer to an inner property of an object value,
     *   use `.` as a separator; in case of conflict, the longest starting substring wins.
     * @param onError - Called in case of error.
     *   If not set, errors are by default logged as warnings.
     */
    formatToParts(msgParams?: Record<string, unknown>, onError?: (error: unknown) => void): MessagePart<P>[];
}
