/**
 * The contents of this file are subject to the OpenXML Public
 * License Version 1.0; you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.openxml.org/license/
 *
 * THIS SOFTWARE AND DOCUMENTATION IS PROVIDED ON AN "AS IS" BASIS
 * WITHOUT WARRANTY OF ANY KIND EITHER EXPRESSED OR IMPLIED,
 * INCLUDING AND WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE
 * AND DOCUMENTATION IS FREE OF DEFECTS, MERCHANTABLE, FIT FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGING. SEE THE LICENSE FOR THE
 * SPECIFIC LANGUAGE GOVERNING RIGHTS AND LIMITATIONS UNDER THE
 * LICENSE.
 *
 * The Initial Developer of this code under the License is
 * OpenXML.org. Portions created by OpenXML.org and/or Assaf Arkin
 * are Copyright (C) 1998, 1999 OpenXML.org. All Rights Reserved.
 */


package examples;


import java.io.*;
import java.util.*;
import org.w3c.dom.*;
import org.w3c.dom.html.*;
import org.openxml.*;
import org.openxml.io.*;
import org.openxml.x3p.*;
import org.openxml.util.*;


/**
 * This example demonstrates parsing an HTML document, getting HTML
 * specific elements using the DOM, publishing the document in HTML
 * and XHTML formats, 
 * 
 * 
 * @version $Revision: 1.2 $ $Date: 1999/04/18 01:48:29 $
 * @author <a href="mailto:arkin@trendline.co.il">Assaf Arkin</a>
 */
public class Ex3
{
    
    
    public static void main( String args[] )
    {
        try
        {
            Parser          parser;
            InputStream     is;
            HTMLDocument    doc;
            Publisher       pub;
            StreamFormat    format;

            // Open the document file, create an HTML parser and parse
            // the file into an HTMLDocument.
            is = new FileInputStream( "examples/ex3.html" );
            parser = DOMFactory.createParser( is, null, DOMFactory.DOCUMENT_HTML );
            doc = (HTMLDocument) parser.parseDocument();

            // If there was any error parsing the document, report it.
            if ( parser.getErrorReport() != null )
                dumpParseErrors( parser.getErrorReport() );
            
            
            // List the contents of the HEAD element followed by the BODY
            // element.
            System.out.println( "Contents of HEAD element:" );
            list( doc.getElementsByTagName( "HEAD" ).item( 0 ), 0 );
            System.out.println( "Contents of BODY element:" );
            list( doc.getElementsByTagName( "BODY" ).item( 0 ), 0 );
            
            // Publish the document in HTML format (output.html) and XHTML
            // format (output.xhtml). Used pretty printing in both case so
            // the output is readable.
            pub = PublisherFactory.createPublisher( new File( "examples/output.html" ), StreamFormat.HTML_PRETTY );
            pub.publish( doc );
            pub.close();
            pub = PublisherFactory.createPublisher( new File( "examples/output.xhtml" ), StreamFormat.XHTML_PRETTY );
            pub.publish( doc );
            pub.close();
            
        }
        catch ( Exception except )
        {
            System.out.println( except );
            except.printStackTrace( System.err );
        }
    }
    
    
    /**
     * Dumps a list of errors generated by the parser. The {@link
     * ErrorReport} object maintained by the parser is used for this
     * purpose. This method will only print errors if there are any.
     * and will print all of them including location information.
     * 
     * @param report The parser's error report
     */
    static void dumpParseErrors( ErrorReport report )
    {
        String[]    messages;
        int         i;
        
        if ( report.getCount() > 0 )
        {
            System.out.println( "Document parsing failed, encountered " + report.getCount() + " error(s)" );
            messages = report.listMessages( ErrorReport.WARNING );
            for ( i = 0 ; i < messages.length ; ++i )
                System.out.println( messages[ i ] );
        }
    }


    /**
     * Lists the contents of a particular node in the document.
     * That would basically be the node's name (for some nodes, type)
     * and its value (for some nodes, empty). If the node has children,
     * they are printed as well. The <tt>level</tt> argument is used
     * for nesting level; when first calling this method, use zero.
     * 
     * @param node The node to list
     * @param level The nesting level
     */
    static void list( Node node, int level )
    {
        int i;
        
        while ( node != null )
        {
            for ( i = 0 ; i < level ; ++i )
                System.out.print( ".." );
            System.out.println( "Node [" + node.getNodeName() + "] = [" + node.getNodeValue() + "]" );
            if ( node.hasChildNodes() )
                list( node.getFirstChild(), level );
            node = node.getNextSibling();
        }
    }


}
