/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.adaptor.rmi.jrmp;

import java.rmi.NoSuchObjectException;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;

import javax.management.JMException;
import javax.management.ObjectName;
import javax.naming.NamingException;

import mx4j.tools.adaptor.rmi.RMIAdaptor;
import mx4j.tools.adaptor.rmi.SSLAdaptorRMIServerSocketFactory;
import mx4j.tools.adaptor.rmi.SSLRMIClientSocketFactory;

/**
 * Adaptor for the JRMP protocol, the native protocol of Java RMI.
 *
 * @author <a href="mailto:biorn_steedom@users.sourceforge.net">Simone Bordet</a>
 * @version $Revision: 1.1 $
 */
public class JRMPAdaptor extends RMIAdaptor implements JRMPAdaptorMBean
{
	private boolean m_running;
	private int m_port;
	private String m_sslFactory;

	/**
	 * Creates a new JRMP adaptor
	 */
	public JRMPAdaptor()
	{
	}

	public void start() throws RemoteException, NamingException, JMException
	{
		if (!isRunning())
		{
			installInterceptors();

			String sslFactory = getSSLFactory();
			if (sslFactory != null)
			{
				ObjectName factory = new ObjectName(sslFactory);
				UnicastRemoteObject.exportObject(this, getPort(), new SSLRMIClientSocketFactory(), new SSLAdaptorRMIServerSocketFactory(getMBeanServer(), factory));
			}
			else
			{
				UnicastRemoteObject.exportObject(this, getPort());
			}

			bind(this);

			m_running = true;
		}
	}

	public void stop() throws NoSuchObjectException, NamingException
	{
		if (isRunning())
		{
			UnicastRemoteObject.unexportObject(this, true);

			unbind();

			uninstallInterceptors();

			m_running = false;
		}
	}

	public boolean isRunning()
	{
		return m_running;
	}

	public String getProtocol()
	{
		return "rmi/jrmp";
	}

	/**
	 * Returns the RMI port on which the adaptor's remote object is exported to
	 * @see #setPort
	 */
	public int getPort()
	{
		return m_port;
	}

	/**
	 * Sets the RMI port on which the adaptor's remote object will be exported to. <br>
	 * This method can be called only if this adaptor is not running.
	 * @see #getPort
	 */
	public void setPort(int port)
	{
		if (!isRunning())
		{
			m_port = port;
		}
		else
		{
			throw new IllegalStateException("Cannot perform this operation while running");
		}
	}

	/**
	 * Returns the ObjectName of the SSL ServerSocket factory MBean used by this adaptor for secure invocations.
	 */
	public String getSSLFactory()
	{
		return m_sslFactory;
	}

	/**
	 * Sets the ObjectName of the SSL ServerSocket factory MBean that will be used by this adaptor for secure invocations. <br>
	 * This method can be called only if this adaptor is not running.
	 */
	public void setSSLFactory(String factory)
	{
		if (!isRunning())
		{
			m_sslFactory = factory;
		}
		else
		{
			throw new IllegalStateException("Cannot perform this operation while running");
		}
	}
}
