/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.naming;

import java.rmi.NoSuchObjectException;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;
import java.rmi.registry.Registry;
import java.rmi.server.UnicastRemoteObject;

/**
 * An MBean that wraps rmiregistry. <p>
 * Calling {@link #start} will launch rmiregistry in the same JVM; this way
 * rmiregistry will have in its classpath the same classes the JVM has.
 *
 * @author <a href="mailto:biorn_steedom@users.sourceforge.net">Simone Bordet</a>
 * @version $Revision: 1.8 $
 */
public class NamingService implements NamingServiceMBean
{
	private int m_port;
	private Remote m_registry;
	private boolean m_running;

	/**
	 * Creates a new instance of NamingService with the default rmiregistry port (1099).
	 */
	public NamingService()
	{
		this(Registry.REGISTRY_PORT);
	}

	/**
	 * Creates a new instance of NamingService with the specified port.
	 */
	public NamingService(int port)
	{
		m_port = port;
	}

	/**
	 * Sets the port on which rmiregistry listens for incoming connections.
	 * @see #getPort
	 */
	public void setPort(int port)
	{
		m_port = port;
	}

	/**
	 * Returns the port on which rmiregistry listens for incoming connections
	 * @see #setPort
	 */
	public int getPort()
	{
		return m_port;
	}

	/**
	 * Returns whether this MBean has been started and not yet stopped.
	 * @see #start
	 */
	public boolean isRunning()
	{
		return m_running;
	}

	/**
	 * Starts this MBean: rmiregistry can now accept incoming calls
	 * @see #stop
	 * @see #isRunning
	 */
	public void start() throws RemoteException
	{
		if (!isRunning())
		{
			m_registry = LocateRegistry.createRegistry(getPort());
			m_running = true;
		}
	}

	/**
	 * Stops this MBean: rmiregistry cannot accept anymore incoming calls
	 * @see #start
	 */
	public void stop() throws NoSuchObjectException
	{
		if (isRunning())
		{
			m_running = !UnicastRemoteObject.unexportObject(m_registry, true);
		}
	}
}
