(*  Copyright (c) 2001 Anthony L Shipman *)


(*  This is common code for timing functions.
*)

signature TIMING =
sig
    val timeIt:	string -> (unit -> 'a) -> 'a
    val report: unit -> unit
end


structure Timing: TIMING =
struct

    (*	To minimise the measurement overhead the entries are 
	saved here. There is no thread-safety but I'll probably
	get away with it given the way the CML scheduling works.
    *)
    type TimeEntry = string * Time.time

    val	queue: TimeEntry list ref = ref []


    fun timeIt name func =
    let
	val now = Time.now()
    in
	func() before
	    let
		val diff = Time.-(Time.now(), now)
	    in
		queue := (name, diff) :: (!queue)
	    end
    end


    fun report() =
    let
	fun show (name, diff) =
	(
	    print(concat["Timing ", name, " ",
		LargeInt.toString(Time.toMicroseconds diff), "\n"])
	)
    in
	app show (rev(!queue))
    end

end



(*  This provides coarser CPU based timing in milliseconds with 10ms resolution.
*)

signature CPU_TIMING =
sig
    val timeIt:	string -> (unit -> 'a) -> 'a
end


structure CPUTiming =
struct

    fun timeIt name func =
    let
	val timer = Timer.startCPUTimer()
    in
	func() before
	    let
		val {usr, sys, gc} = Timer.checkCPUTimer timer
	    in
		print(concat["CPU Timing ", name, " usr=",
		    LargeInt.toString(Time.toMilliseconds usr),
		    "ms GC=",
		    LargeInt.toString(Time.toMilliseconds gc),
		    "ms\n"])
	    end
    end

end

