(*  Copyright (c) 2001 Anthony L Shipman *)


structure Main=
struct

    fun toErr s = TextIO.output(TextIO.stdErr, s)

    structure BT  = RegExpFn(structure P=AwkSyntax
		             structure E=BackTrackEngine)

    structure DFA = RegExpFn(structure P=AwkSyntax
		             structure E=BackTrackEngine)

    fun demo1BT msg =
    let
	val regexp = BT.compileString "the.(quick|slow).brown"
    in
	case StringCvt.scanString (BT.find regexp) msg of
	  NONE      => print "demo1 match failed\n"
	| SOME tree => show_matches msg tree
    end


    and demo1DFA msg =
    let
	val regexp = DFA.compileString "the.(quick|slow).brown"
    in
	case StringCvt.scanString (DFA.find regexp) msg of
	  NONE      => print "demo1 match failed\n"
	| SOME tree => show_matches msg tree
    end


    (*  Show the matches n=0, ... *)
    and show_matches msg tree =
    let
	val last = MatchTree.num tree

	fun find n =
	(
	    case MatchTree.nth(tree, n) of
	      NONE => "<Unmatched>"

	    | SOME {pos, len} => String.substring(msg, pos, len)
	)

	and loop n =
	(
	    print(concat[Int.toString n, " => ", find n, "\n"]);
	    if n >= last then () else loop(n+1)
	)
    in
	loop 0
    end



    local
	structure RE = RegExpSyntax
	structure CS = RE.CharSet

	val dot = RE.NonmatchSet(CS.singleton #"\n")

	fun cvt_str s = RE.Concat(map RE.Char (explode s))
    in
	fun demo2BT msg =
	let
	    (* "the.(quick|slow).brown" *)
	    val regexp = BackTrackEngine.compile(RE.Concat[
	    		    cvt_str "the",
			    dot,
			    RE.Group(RE.Alt[
				cvt_str "quick",
				cvt_str "slow"]),
			    dot,
			    cvt_str "brown"
			    ])
	in
	    case StringCvt.scanString (BT.find regexp) msg of
	      NONE      => print "demo2 match failed\n"
	    | SOME tree => show_matches msg tree
	end
    end


    fun main(arg0, argv) =
    let
	val msg = "the quick brown fox"
    in
	print "Demo 1 using BT\n";
	demo1BT msg;

	print "\n";
	print "Demo 1 using DFA\n";
	demo1DFA msg;

	print "\n";
	print "Demo 1 failure using BT\n";
	demo1BT "no quick brown fox";

	print "\n";
	print "Demo 2 using BT\n";
	demo2BT "the quick brown fox";

	OS.Process.success
    end
    handle x =>
    (
	toErr(exnMessage x); toErr("\n");
	OS.Process.failure
    )

    val _ = SMLofNJ.exportFn("regexp", main)
end
