(*  Copyright (c) 2001 Anthony L Shipman *)


(*  This experiments with delivering time-out events using an Ivar.
    The goal is to have one manager thread that does the timing. It
    broadcasts an event to all timed-out threads.

*)

structure Main =
struct
    fun toErr msg = TextIO.output(TextIO.stdErr, msg)


    (*	The number of thousands of threads to spawn. *)
    val M = 1

    fun toTime n = Time.fromSeconds(LargeInt.fromInt n)

(*===================================================================*)


    fun thread n evt () =
    let
	fun report() = print(concat["Thread ", Int.toString n,
			" finishes at ",
			Time.fmt 6 (Time.now()), "\n"])
    in
	CML.sync(CML.wrap(evt, report))
    end
    handle x => toErr(concat["Uncaught exception: ", exnMessage x,
    			     " from thread ", Int.toString n, "\n"])



    (*	This models the behaviour of the web server which creates a time-out
	event and then spawns a thread to wait for it.
    *)
    fun spawner evt 0 = ()
    |   spawner evt n =
    let
	val name = concat["Thread ", Int.toString n]
    in
	Timing.timeIt name (fn() => ignore(CML.spawn (thread n evt)));
	spawner evt (n-1)
    end


    fun run arg () =
    let
	val evt = 
	    case arg of
	      "time" => CML.atTimeEvt(Time.+(Time.now(), toTime(M * 5)))
	    | _      => CML.timeOutEvt(toTime(M * 5))
    in
	spawner evt (M * 1000);
	Timing.report()
    end



    fun main(arg0, argv) =
    let
	val arg = if null argv then "" else hd argv
    in
	RunCML.doit(run arg, NONE);
        OS.Process.success
    end
    handle
      x =>
    (
	toErr(concat["Uncaught exception: ", exnMessage x, " from\n"]);
	app (fn s => (print "\t"; print s; print "\n")) (SMLofNJ.exnHistory x);
	OS.Process.failure
    )

    val _ = SMLofNJ.exportFn("timeout_evt", main)
end


