(*  Copyright (c) 2001 Anthony L Shipman *)

(* $Id: file_io.sml,v 1.13 2001/09/10 20:45:31 felix Exp $ *)

(*  This contains functions for operating on files.
    This will log errors.

*)

signature FILEIO =
sig

    (*	This removes a file.  It logs errors and raises no exceptions.
	It is not an error if the file does not exist.
	The path should be absolute.
    *)
    val removeFile: string -> unit


    (*	This creates a file while checking for name clashes.
	It returns true if successful, false if there was a name clash,
	or logs an error and raises an exception for all other errors.

	This does not cope with NFS-mounted file systems.
    *)
    val exclCreate: string -> bool


    (*	Return the size of the file in bytes or NONE if the file can't be
	accessed.
    *)
    val fileSize:   string -> int option


    (*	Return the last modification time of the file or NONE if the file
	doesn't exist.
    *)
    val modTime:   string -> Time.time option


    (*	Set a socket to close-on-exec. 
    *)
    val setCloseExec:    OS.IO.poll_desc -> unit


    (*	This handles the opening and closing of an input text file.
	Exceptions are carefully handled. The default value is returned
	if an exception happens.

	This does not use the OpenMgr since it uses TextIO streams
	directly and callers will want to use the inputLine function.
	This should just be used during startup for reading configuration
	files.

    *)
    val withTextIn: 
		    Abort.Abort ->
		    string ->
     		    'a ->		(* the default *)
		    (TextIO.instream -> 'a) ->
		    'a


    (*	This lists the files in a directory. The names are relative to
	the given directory. The . and .. directories are not included.
	This is intended for indexing directories.  The files are in no
	particular order.

	This logs and raises an exception if the listing failed
	e.g. because the directory does not exist or is not accessible.

	This uses OpenMgr so it is fine for use within connections.
	The first arg is an abort event.
    *)
    val listDir:    Abort.Abort -> string -> string list

end



structure FileIO: FILEIO =
struct
    open Common

    structure TF  = TextFrag
    structure FS  = Posix.FileSys
    structure IO  = Posix.IO

(*------------------------------------------------------------------------------*)



    fun removeFile file =
    (
	if Files.exists file
	then
	    OS.FileSys.remove file
	else
	    ()
    )
    handle x => Log.logExnArg file x



    fun exclCreate file =
    (
	IO.close(FS.createf(file, FS.O_WRONLY, FS.O.excl,
	           FS.S.flags[FS.S.irusr, FS.S.iwusr]));
	true
    )
    handle
      x as OS.SysErr (_, eopt) =>
	(if isSome eopt andalso valOf eopt = Posix.Error.exist
	 then 
	    false	(* failed to exclusively create *)
	 else
	    (Log.logExnArg file x; raise x) (* failed with error *)
	)

    | x => (Log.logExnArg file x; raise x)



    fun fileSize file = 
    (
	SOME(OS.FileSys.fileSize file)
    )
    handle x => (Log.logExnArg file x; NONE)



    fun modTime file = 
    (
	SOME(OS.FileSys.modTime file)
    )
    handle x => (Log.logExnArg file x; NONE)



    fun setCloseExec poll_desc =
    let
	val fd = valOf(FS.iodToFD(OS.IO.pollToIODesc poll_desc))
    in
	IO.setfd(fd, IO.FD.cloexec)
    end


    fun withTextIn abort file default func =
    let
    in
	case TextIOReader.openIt abort file of
	  NONE   => default		  (* the open failed *)
	| SOME h =>
	    ((func (TextIOReader.get h))  (* handle an I/O failure with closing *)
		handle x => (Log.logExn x; TextIOReader.closeIt h; default)
	    ) before (TextIOReader.closeIt h)
    end
    handle x => (Log.logExnArg file x; default) 




    fun listDir abort dir =
    let
	fun loop strm rslt =
	(
	    case OS.FileSys.readDir strm of
	      "" => rslt
	    | s  => loop strm (s::rslt)
	)
    in
	case DirReader.openIt abort dir of
	  NONE => []		    (* the open failed *)

	| SOME h =>
	    (loop (DirReader.get h) []) before (DirReader.closeIt h)
    end
    handle x => (Log.logExnArg dir x; raise x) 

(*------------------------------------------------------------------------------*)

end
