(*  Copyright (c) 2001 Anthony L Shipman *)

(* $Id: logging.sml,v 1.22 2002/01/19 16:01:42 felix Exp $ *)

(*  This contains the logging facility.  It can write to stderr if
    no logging file has been specified.

    The facility maintains a buffer of log messages that are written to
    the log file concurrently. This avoids holding up the server to log
    messages and avoids races.  I'll use a Mailbox for the buffer and
    not worry about it being unbounded.

    I'll have to flush the log file periodically get the messages out
    promptly.  A one second flush interval should be fine.

@#34567890123456789012345678901234567890123456789012345678901234567890
*)

signature LOG =
sig

    type Level

    val Fatal: Level
    val Error: Level
    val Warn:  Level
    val Info:  Level
    val Debug: Level

    (*	This writes a logging message.
	Error messages are counted.
    *)
    val	log:	    Level -> TextFrag.Text -> unit

    (*	This writes a logging message related to some source file position.
	Error messages are counted.
    *)
    val	logP:	    Level -> Common.SrcPos -> TextFrag.Text -> unit


    (*	Shorthands. *)
    val	fatal:	    string list -> unit
    val	error:	    string list -> unit
    val	errorP:	    Common.SrcPos -> string list -> unit

    (*	This is for informational messages where we don't want to
	evaluate the message much if the message won't be logged. So
	we delay the evaluation with a function wrapper.
    *)
    val inform:	    Level -> (unit -> TextFrag.Text) -> unit
    val testInform: int -> Level -> (unit -> TextFrag.Text) -> unit


    (*	This returns the count of error messages seen so far.
    *)
    val	numErrors:  unit -> int

    (*	This waits until the log messages have drained.
    *)
    val flush:	    unit -> unit

    (*	Set the file for error logging.  This is currently only
	allowed to be set once.
    *)
    val setLogFile: string -> unit

    (*	Set the level for error logging. *)
    val setLevel:   Level -> unit

    (*	Set the level for error logging to be at least as low as the
	given level.
    *)
    val lowerLevel:   Level -> unit


    (*	Describe the level for printing. *)
    val formatLevel:	Level -> string

    (*-------------------------------------*)

    (*	Log any kind of exception.
	This is guaranteed not to raise any exception.
    *)
    val logExn:  exn -> unit

    (*	Log with some extra information e.g. a file name. *)
    val logExnArg:  string -> exn -> unit

end



structure Log: LOG =
struct

    open Common
    structure Sy = SyncVar
    structure TF = TextFrag

(*------------------------------------------------------------------------------*)

    type Level = int

    (*	Higher levels are more important. *)
    val Fatal: Level	= 8
    val Error: Level	= 7
    val Warn:  Level	= 6
    val Info:  Level	= 5
    val Debug: Level	= 4

    (*	This is the current log level. *)
    val log_level = ref Error

    fun formatLevel 8 = "Fatal"
    |   formatLevel 7 = "Error"
    |   formatLevel 6 = "Warning"
    |   formatLevel 5 = "Info"
    |   formatLevel 4 = "Debug"
    |   formatLevel _ = "?"

(*------------------------------------------------------------------------------*)

    (*	The internal logging protocol. *)

    datatype LogRequest =
	    ReqLog of Level * TF.Text * Time.time
	|   ReqSetFile of string
	|   ReqNumErrors of int Sy.ivar
	|   ReqFlush of unit Sy.ivar	    (* flush with acknowledge *)



    fun log_server mbox () =
    let
	(*  An imperative state will be OK in this small context. *)
	val num_errors: int ref = ref 0
	val log_strm  = ref TextIO.stdErr
	val log_file  = ref ""


	fun loop() =
	let
	    fun timeout() = TextIO.flushOut(!log_strm)
	in
	    CML.select[
		CML.wrap(Mailbox.recvEvt mbox, handle_msg),
		CML.wrap(CML.timeOutEvt (Time.fromSeconds 1), timeout)
		];
	    loop()
	end


	and handle_msg msg =
	(
	    case msg of
	      ReqLog (level, msg, time) => internal_log level msg time

	    | ReqSetFile file =>
		(
		    if !log_file = ""
		    then
			set_log_file file
		    else
			internal_log Error (TF.S
			    "Attempted to set the log file twice")
			    (Time.now())
		)

	    | ReqNumErrors rvar => Sy.iPut(rvar, !num_errors)

	    | ReqFlush rvar => Sy.iPut(rvar, ())
	)


	and internal_log level msg when =
	let
	    fun put s = TextIO.output(!log_strm, s)

	    val date  = Date.fromTimeLocal(when)
	    val fdate = Date.fmt "%Y %b %d %H:%M:%S" date
	in
	    put(concat[fdate, " ", formatLevel level, ": "]);
	    TF.appPrefix "\t" put msg;
	    put "\n";
	    update_counts level
	end


	and update_counts level =
	(
	    if level >= Error
	    then
		num_errors := !num_errors + 1
	    else
		()
	)


	and set_log_file file =
	(
	    log_strm := TextIO.openAppend file;
	    log_file := file
	)
	handle IO.Io {name, function, cause} =>
	(
	    toErr(concat["IO Error ", name,
			 " ", function,
			 " ", exnMessage cause,
			 "\n"])
	)

    in
	loop()
    end

(*------------------------------------------------------------------------------*)

    structure Logger = Singleton(
    			    type input    = LogRequest Mailbox.mbox
			    val  newInput = Mailbox.mailbox
			    val  object   = log_server
			    )

    fun	send_request req = Mailbox.send (Logger.get(), req)


    fun flush() =
    let
	val rvar = Sy.iVar()
    in
	send_request (ReqFlush rvar);
	Sy.iGet rvar
    end


    fun	log level msg = 
    (
	if level >= (!log_level) orelse level = Fatal (* speed up check *)
	then
	    send_request (ReqLog (level, msg, Time.now()))
	else
	    ();

	if level = Fatal
	then
	(
	    flush();
	    Common.toErr(concat[formatLevel level, ": ",
	    			TF.toString TF.UseLf msg, "\n"]);
	    Common.fail()		(* abandon execution *)
	)
	else
	    ()
    )


    (*	This writes a logging message related to some source file position.
	Error messages are counted.
    *)
    fun	logP level pos msg = log level (TF.C [TF.S(formatPos pos), TF.S ": ", msg])
    

    fun	fatal msg  = log Fatal (TF.L msg)
    fun	error msg  = log Error (TF.L msg)
    fun	errorP pos msg = logP Error pos (TF.L msg)


    fun inform level msg_func =
    (
	if level >= (!log_level)	(* speed up the check *)
	then
	    send_request (ReqLog (level, msg_func(), Time.now()))
	else
	    ()
    )


    fun testInform test level msg_func =
    (
	if Globals.testing test
	then
	    inform level msg_func
	else
	    ()
    )


    (*	This returns the count of error messages seen so far.
    *)
    fun	numErrors() =
    let
	val rvar = Sy.iVar()
    in
	send_request (ReqNumErrors rvar);
	Sy.iGet rvar
    end

    fun setLogFile file = send_request (ReqSetFile file)

    fun setLevel   level = log_level := level
    fun lowerLevel level = log_level := Int.min(level, !log_level)

(*------------------------------------------------------------------------------*)

    (*	Format the error from an OS.SysErr exception in a
	standard way.

	Fatal and InternalErrors always abort the server.
    *)
    fun logSysErr arg (msg: string, _: OS.syserror option) = error [arg, " ", msg]

    fun logExn x = logExnArg "" x

    and logExnArg arg x =
    (
	case x of
	  OS.SysErr x        => logSysErr arg x
	| IO.Io {cause, ...} => logExnArg arg cause

	| InternalError msg  => log Fatal (TF.L ["Internal Error: ", msg])
	| FatalX             => log Fatal (TF.S "Fatal Error")

	| x                  => log_any x
    )
    handle _ => ()		(* don't raise any more exceptions *)

    and log_any x = log Error (TF.L [exnName x, ": ", exnMessage x])

(*-------------------------------------------------------------------*)

end
