(*  Copyright (c) 2001 Anthony L Shipman *)

(* $Id: builtin_node.sml,v 1.11 2001/08/20 16:42:00 felix Exp $ *)

(*  

    Simple built-ins have no state. They just generate a canned response.
    The kind of reponse is determined by the configuration.

@#34567890123456789012345678901234567890123456789012345678901234567890
*)


structure SimpleBuiltinHandler: NODE_HANDLER =
struct

    open Common

    structure M      = Mailbox
    structure Cfg    = Config
    structure Req    = HTTPMsg
    structure Status = HTTPStatus
    structure U      = ResponseUtils
    structure TF     = TextFrag

    open Node

(*------------------------------------------------------------------------------*)

    type CreateArg  = string

    fun canTakeLast _ = false
    fun canTakeRest _ = false

    
    (*	We check the name is legal here. REVISIT It
	should be checked when the main config is read.
    *)
    fun init name =
    let
	val mbox  = M.mailbox()
    in
	case get_maker name of
	  NONE   => Log.error ["Invalid built-in handler name: ", name]
	| SOME _ => ();

	CML.spawn (server mbox);

	(mbox, NONE)
    end



    and server mbox () =
    let
	fun loop() =
	(
	    case M.recv mbox of
	      msg as HndReq _ => handle_request msg 

	    ; loop()
	)
    in
	loop()
    end



    and handle_request 
	    (msg as HndReq {config, rchan, request, ...})
	    =
    let
	val Cfg.NodeConfig {kind, ...} = config

	fun reply response =
	(
	    CML.send(rchan, HndResponse(msg, response))
	)
    in
	case kind of
	  Cfg.NodeIsBuiltin {name} =>
	(
	    case get_maker name of
	      NONE   => reply (U.mkServerFail())
	    | SOME f => reply (f request)
	)

	| _ => raise InternalError "SimpleBuiltin,handleRequest"
    end



    and get_maker name =
    (
	case name of
	  "hw"	    => SOME (fn _ => U.mkHelloWorld())
	| "reject"  => SOME (fn _ => U.mkNotFound())
	| "sleep"   => SOME sleep
	| _         => NONE
    )




    (*	This sleeps for a number of seconds given by the time-out
	value in the query.  It is useful for holding a connection
	open while testing. At the moment we can't run more than one of
	these concurrently.

	Don't forget to abort the sleep if the connection goes down.
    *)
    and sleep request =
    let
	val Req.Request  {url, abort, ...} = request
	val URL.HTTP_URL {query, ...} = url

	val timeout = 
	    case query of
	      NONE => 1
	    | SOME q => getOpt(Int.fromString q, 1)

	val t_evt = CML.timeOutEvt(Time.fromSeconds(Int.toLarge timeout))
    in
	CML.select[
	    CML.wrap(t_evt, fn _ => ()),
	    CML.wrap(Abort.evt abort, fn _ => ())
	    ];

	U.mkHTML Status.OK (TF.L [
		"<html><body><p>", 
		"Slept for ", Int.toString timeout, " seconds",
		"</body></html>"])
    end


(*------------------------------------------------------------------------------*)

end
